﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Windows.Forms;

using Inet.Viewer.Data;
using Inet.Viewer.Resources;
using System.Text.RegularExpressions;

namespace Inet.Viewer.WinForms.Export
{
    /// <summary>
    /// Dialog for exporting a report where the user can set the output file and
    /// format. Each format can have a individual settings page for configuring
    /// format-specific options. <br/>
    /// Before opening the dialog, the properties ReportData and ReportInfo must be
    /// set.
    /// </summary>
    public partial class ExportDialog : Form
    {
        private readonly FormatSettingsControl[] settingControls = {
            new PDFSettingsControl(),
            new PSSettingsControl(),
            new CSVSettingsControl(),
            new ExcelSettingsControl(),
            new ODSSettingsControl(),
            new HTMLSettingsControl(),
            new XMLSettingsControl(),
            new TextSettingsControl(),
            new ImageSettingsControl(),
            new RTFSettingsControl(),
            new SVGSettingsControl(),
        };

        private SaveFileDialog fileDialog = new SaveFileDialog();
        private bool formatSelectionSubControl = true;
        private FormatSelectControl formatSelectControl;

        /// <summary>
        /// Creates the export dialog.
        /// </summary>
        public ExportDialog()
        {
            formatSelectControl = new FormatSelectControl(settingControls);
            InitializeComponent();
            fileDialog.OverwritePrompt = false;
            btnConfigSwitch.Text = strings.ExportFormatSettings;
            formatSelectControl.FormatChanged += new EventHandler(OnFormatChanged);
            tbFile.Text = Directory.GetCurrentDirectory() + "\\unnamed";
            SetSelectionMode(true);
            foreach (FormatSettingsControl settingsControl in settingControls)
            {
                settingsControl.FormatChanged += OnFormatChanged;
            }
        }

        /// <summary>
        /// Creates a dictionary with the export parameters choosen by the user.
        /// </summary>
        /// <returns>the export parameters</returns>
        public Dictionary<string, string> CreateExportParameters()
        {
            Dictionary<string, string> exportParams = new Dictionary<string, string>();
            exportParams[URLRenderData.ParameterFile] = tbFile.Text;
            formatSelectControl.SelectedFormatSettingsControl.CollectExportParameters(exportParams);

            if (cbOpenFile.Checked)
            {
                exportParams["exportInApplication"] = "true";
            }

            return exportParams;
        }

        /// <summary>
        /// Flag whether the report is a multi-page report (when rendering with the viewer).
        /// </summary>
        public bool MultiPageReport
        {
            set
            {
                // forward the report data to each format settings control:
                foreach (FormatSettingsControl settingsControl in settingControls)
                {
                    settingsControl.MultiPageReport = value;
                }
            }
        }

        /// <summary>
        /// Sets a flag indicating whether the report has any groups.
        /// </summary>
        public bool HasGroups
        {
            set
            {
                // forward the report data to each format settings control:
                foreach (FormatSettingsControl settingsControl in settingControls)
                {
                    settingsControl.HasGroups = value;
                }
            }
        }

        /// <summary>
        /// Report info. Must be set before opening the dialog.
        /// </summary>
        public ReportInfo ReportInfo
        {
            set
            {
                formatSelectControl.AllowedFormats = value.Formats;
                // replace the file name with the report title
                var sel = formatSelectControl.SelectedFormatSettingsControl;
                Regex regExp = new Regex(string.Format("[" + Regex.Escape(new string(Path.GetInvalidFileNameChars()) + new string(Path.GetInvalidPathChars()) + "]")));
                tbFile.Text = Path.GetDirectoryName(tbFile.Text) + "\\" + regExp.Replace(value.Title, "") + "." + (sel == null ? "dat" : sel.FileSuffix);
            }
        }

        /// <summary>
        /// Called when the user clicks on the output file button. Opens a file dialog
        /// for selection.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnFileDialog_Click(object sender, EventArgs e)
        {
            fileDialog.FileName = tbFile.Text;
            fileDialog.Filter = formatSelectControl.SelectedFormatSettingsControl.Label + " Files | *." + formatSelectControl.SelectedFormatSettingsControl.FileSuffix;
            if (fileDialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                tbFile.Text = fileDialog.FileName;
            }
        }

        /// <summary>
        /// Called when the users clicks on the config switch button. Toggles between
        /// the format selection panel and the format settings panel.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnConfigSwitch_Click(object sender, EventArgs e)
        {
            SetSelectionMode(!formatSelectionSubControl);
        }

        /// <summary>
        /// Sets whether the selection mode is active. In selection mode a list 
        /// of formats is shown, otherwise the format specific page is shown.
        /// </summary>
        /// <param name="formatSelectionSubControl">flag indicating the selection mode</param>
        private void SetSelectionMode(bool formatSelectionSubControl)
        {
            this.formatSelectionSubControl = formatSelectionSubControl;
            UserControl uc;
            if (formatSelectionSubControl)
            {
                uc = formatSelectControl;
                btnConfigSwitch.Text = strings.ExportFormatSettings;
                this.btnConfigSwitch.Image = global::Inet.Viewer.Images.DataContainer_MoveNextHS;
            }
            else
            {
                uc = formatSelectControl.SelectedFormatSettingsControl;
                btnConfigSwitch.Text = strings.ExportFormatSelection;
                this.btnConfigSwitch.Image = global::Inet.Viewer.Images.DataContainer_MovePreviousHS;
            }

            uc.Dock = DockStyle.Fill;
            panel.Controls.Clear();
            panel.Controls.Add(uc);
        }

        /// <summary>
        /// Called when the cancel button was clicked. Closes the dialog.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnCancel_Click(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
            Close();
        }

        /// <summary>
        /// Called when the export button was clicked. Closes the dialog with OK 
        /// as result.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnExport_Click(object sender, EventArgs e)
        {
            if (File.Exists(tbFile.Text) && (MessageBox.Show(string.Format(Inet.Viewer.Resources.strings.ConfirmFileOverwrite, tbFile.Text), Inet.Viewer.Resources.strings.ConfirmFileOverwriteTitle, MessageBoxButtons.YesNo) != DialogResult.Yes))
            {
                return;   
            }
            DialogResult = DialogResult.OK;
            Close();
        }

        /// <summary>
        /// Called when the format changed. Updates the file name and the config switch
        /// button state.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void OnFormatChanged(object sender, EventArgs e)
        {
            if (formatSelectControl.SelectedFormatSettingsControl != null)
            {
                tbFile.Text = Path.GetDirectoryName(tbFile.Text) + "\\" + Path.GetFileNameWithoutExtension(Path.GetFileNameWithoutExtension(tbFile.Text)) +
                              "." + formatSelectControl.SelectedFormatSettingsControl.FileSuffix;
                btnConfigSwitch.Enabled = formatSelectControl.SelectedFormatSettingsControl.HasChildren && formatSelectControl.SelectedFormatSettingsControl.Enabled;
            }
        }

        /// <summary>
        /// Called when the user opening the dialog. Shows the format selection.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void exportDialog_Load(object sender, EventArgs e)
        {
            SetSelectionMode(true);
        }


        /// <summary>
        /// A panel with a self-drawn background.
        /// </summary>
        public class BottomPanel : System.Windows.Forms.Panel
        {
            /// <summary>
            /// Creates the panel, sets the style.
            /// </summary>
            public BottomPanel()
            {
                SetStyle(System.Windows.Forms.ControlStyles.UserPaint, true);
            }
            /// <summary>
            /// <inheritdoc/>
            /// </summary>
            protected override void OnPaintBackground(PaintEventArgs e)
            {
                e.Graphics.FillRectangle(new SolidBrush(System.Drawing.SystemColors.Control), 0, 0, Width, Height);
                GraphicsPath p = new GraphicsPath();
                int w = Width;
                int h = Height;
                int s = 80;
                Point p1 = new Point(0, h);
                Point p2 = new Point(s, h);
                Point p3 = new Point(100 - s, 0);
                Point p4 = new Point(100, 0);
                Point p6 = new Point(w, 0);
                Point p7 = new Point(w, h);
                p.AddBezier(p1, p2, p3, p4);
                p.AddLine(p4, p6);
                p.AddLine(p6, p7);
                p.AddLine(p7, p1);
                e.Graphics.SmoothingMode = SmoothingMode.AntiAlias;
                e.Graphics.FillPath(new SolidBrush(BackColor), p);
                e.Graphics.DrawPath(new Pen(ForeColor), p);
            }
        }

        /// <summary>
        /// Returns a flag indicating whether any format is available to export.
        /// </summary>
        public bool AnyFormatAvailable
        {
            get
            {
                return formatSelectControl.AnyFormatAvailable;
            }
        } 
    }
}
